<?php
// Pastikan tidak ada output sebelumnya
ob_clean();
ob_start();

require_once 'includes/config.php';
require 'vendor/autoload.php';

use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;

// Cek jika user belum login
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit();
}

try {
    // Ambil filter yang aktif dari URL
    $filters = [
        'action' => $_GET['action'] ?? '',
        'marketing_id' => $_GET['marketing'] ?? '',
        'date_from' => $_GET['date_from'] ?? '',
        'date_to' => $_GET['date_to'] ?? ''
    ];

    // Ambil data sesuai filter
    $allData = getMarketingData($filters);

    // Ambil parameter paginasi dari URL
    $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
    $perPage = 10;

    // Gunakan fungsi paginasi yang sama dengan dashboard
    $paginationResult = paginateData($allData, $page, $perPage);
    $data = $paginationResult['data'];

    // Buat spreadsheet baru
    $spreadsheet = new Spreadsheet();
    $sheet = $spreadsheet->getActiveSheet();

    // Set judul kolom
    $sheet->setCellValue('A1', 'No.');
    $sheet->setCellValue('B1', 'Target Number');
    $sheet->setCellValue('C1', 'Action');
    $sheet->setCellValue('D1', 'Marketing');
    $sheet->setCellValue('E1', 'Note');
    $sheet->setCellValue('F1', 'Date');
    $sheet->setCellValue('G1', 'ND Status');

    // Style untuk header
    $headerStyle = [
        'font' => [
            'bold' => true,
        ],
        'fill' => [
            'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
            'startColor' => [
                'rgb' => 'E2E3E5',
            ],
        ],
    ];
    $sheet->getStyle('A1:G1')->applyFromArray($headerStyle);

    // Setelah set header style, tambahkan format khusus untuk kolom nomor telepon
    $sheet->getStyle('B:B')->getNumberFormat()->setFormatCode(\PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_TEXT);

    // Isi data
    $row = 2;
    foreach ($data as $index => $item) {
        $currentNumber = (($page - 1) * $perPage) + $index + 1;
        
        $sheet->setCellValue('A' . $row, $currentNumber);
        // Tambahkan tanda kutip di awal nomor untuk memaksa format teks
        $sheet->setCellValueExplicit(
            'B' . $row, 
            $item['phone_number'],
            \PhpOffice\PhpSpreadsheet\Cell\DataType::TYPE_STRING
        );
        $sheet->setCellValue('C' . $row, $item['status']);
        $sheet->setCellValue('D' . $row, getMarketingName($item['marketing_id']));
        $sheet->setCellValue('E' . $row, $item['note']);
        $sheet->setCellValue('F' . $row, date('d-M-Y', strtotime($item['original_call_date'])));
        $sheet->setCellValue('G' . $row, $item['nd_status'] == 'active' ? 'Yes' : 'No');
        $row++;
    }

    // Auto size kolom
    foreach (range('A', 'G') as $col) {
        $sheet->getColumnDimension($col)->setAutoSize(true);
    }

    // Set style untuk seluruh data
    $styleArray = [
        'borders' => [
            'allBorders' => [
                'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN,
            ],
        ],
    ];
    $sheet->getStyle('A1:G'.($row-1))->applyFromArray($styleArray);

    // Generate nama file dengan timestamp
    $filename = 'marketing_report_page' . $page . '_' . date('Y-m-d_H-i-s') . '.xlsx';

    // Hapus output buffer
    ob_end_clean();

    // Set header untuk download
    header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    header('Cache-Control: max-age=0');
    header('Cache-Control: max-age=1');
    header('Expires: Mon, 26 Jul 1997 05:00:00 GMT');
    header('Last-Modified: ' . gmdate('D, d M Y H:i:s') . ' GMT');
    header('Cache-Control: cache, must-revalidate');
    header('Pragma: public');

    // Tulis ke output
    $writer = new Xlsx($spreadsheet);
    $writer->save('php://output');

} catch (Exception $e) {
    // Log error
    error_log("Error exporting Excel: " . $e->getMessage());
    
    // Redirect kembali dengan pesan error
    $_SESSION['error'] = "Gagal mengexport data. Silakan coba lagi.";
    header("Location: dashboard.php");
}

exit();